//
//  StringRendering.m
//  HelloWorld
//
//  Created by Erica Sadun on 7/29/11.
//  Copyright 2011 Up To No Good, Inc. All rights reserved.
//

#import "StringRendering.h"
#import "StringHelper.h"

@implementation StringRendering
@synthesize string, view, inset;
+ (id) rendererForView: (UIView *) aView string: (NSAttributedString *) aString
{
    StringRendering *renderer = [[self alloc] init];
    renderer.view = aView;
    renderer.string = aString;
    return renderer;
}

// Przygotowanie odwróconego kontekstu.
- (void) prepareContextForCoreText
{
    CGContextRef context = UIGraphicsGetCurrentContext();
	CGContextSetTextMatrix(context, CGAffineTransformIdentity);
	CGContextTranslateCTM(context, 0, view.bounds.size.height);
	CGContextScaleCTM(context, 1.0, -1.0); // Odwrócenie kontekstu.
}

// Dostosowanie wyświetlanego prostokąta, aby zrekompensować odwrócony kontekst.
- (CGRect) adjustedRect: (CGRect) rect
{
    CGRect newRect = rect;
    CGFloat newYOrigin = view.frame.size.height - (rect.size.height + rect.origin.y);
    newRect.origin.y = newYOrigin;
    return newRect;
}

// Dodanie tekstu do prostokąta.
- (void) drawInRect: (CGRect) theRect
{
    CGContextRef context = UIGraphicsGetCurrentContext();
    CGRect insetRect = CGRectInset(theRect, inset, inset);
	CGRect rect = [self adjustedRect: insetRect];
	CGMutablePathRef path = CGPathCreateMutable();
	CGPathAddRect(path, NULL, rect);
    
	CTFramesetterRef framesetter = CTFramesetterCreateWithAttributedString((__bridge CFAttributedStringRef)string);
	CTFrameRef theFrame = CTFramesetterCreateFrame(framesetter, CFRangeMake(0, string.length), path, NULL);
	
	CTFrameDraw(theFrame, context);
	
	CFRelease(framesetter);
	CFRelease(theFrame);
	CFRelease(path);
}

// Wyświetlenie na ścieżce.
- (void) drawInPath: (CGMutablePathRef) path
{
    CGContextRef context = UIGraphicsGetCurrentContext();
	CTFramesetterRef framesetter = CTFramesetterCreateWithAttributedString((__bridge CFAttributedStringRef)string);
	CTFrameRef theFrame = CTFramesetterCreateFrame(framesetter, CFRangeMake(0, string.length), path, NULL);
	
	CTFrameDraw(theFrame, context);
	
	CFRelease(framesetter);
	CFRelease(theFrame);
	CFRelease(path);	

}

// Odległość pomiędzy dwoma punktami.
float distance (CGPoint p1, CGPoint p2)
{
	float dx = p2.x - p1.x;
	float dy = p2.y - p1.y;
	
	return sqrt(dx*dx + dy*dy);
}

- (void) drawOnPoints: (NSArray *) points
{
	int pointCount = points.count;
	if (pointCount < 2) return;
    
    // OBLICZENIA
    
    // Obliczenie długości ścieżki punktów.
	float totalPointLength = 0.0f;
	for (int i = 1; i < pointCount; i++)
		totalPointLength += distance([[points objectAtIndex:i] CGPointValue], [[points objectAtIndex:i-1] CGPointValue]);
	
	// Utworzenie linii typograficznej, pobranie jej długości.
	CTLineRef line = CTLineCreateWithAttributedString((__bridge CFAttributedStringRef)string);
	if (!line) return;
	double lineLength = CTLineGetTypographicBounds(line, NULL, NULL, NULL);
	
	// Pobranie przebiegów.
	CFArrayRef runArray = CTLineGetGlyphRuns(line);
	
	// Zliczenie elementów.
	int glyphCount = 0; //  Zliczenie elementów.
	float runningWidth; //  Szerokość bieżącego znaku.
	int glyphNum = 0;   //  Bieżący znak.
	for (CFIndex runIndex = 0; runIndex < CFArrayGetCount(runArray); runIndex++) 
	{
		CTRunRef run = (CTRunRef)CFArrayGetValueAtIndex(runArray, runIndex);
		for (CFIndex runGlyphIndex = 0; runGlyphIndex < CTRunGetGlyphCount(run); runGlyphIndex++) 
		{
			runningWidth += CTRunGetTypographicBounds(run, CFRangeMake(runGlyphIndex, 1), NULL, NULL, NULL);
			if (!glyphNum && (runningWidth > totalPointLength))
				glyphNum = glyphCount;
			glyphCount++;
		}
	}
    
    // Użycie długości całkowitej do obliczenia procentu ścieżki w danym punkcie.
	NSMutableArray *pointPercentArray = [NSMutableArray array];
	[pointPercentArray addObject:[NSNumber numberWithFloat:0.0f]];
	float distanceTravelled = 0.0f;
	for (int i = 1; i < pointCount; i++)
	{
		distanceTravelled += distance([[points objectAtIndex:i] CGPointValue], [[points objectAtIndex:i-1] CGPointValue]);
		[pointPercentArray addObject:[NSNumber numberWithFloat:(distanceTravelled / totalPointLength)]];
	}
	
	// Dodanie ostatniego elementu. Prawdopodobnie nie będzie potrzebne. 
	[pointPercentArray addObject:[NSNumber numberWithFloat:2.0f]];
    
    
    // PRZYGOTOWANIE DO WYŚWIETLENIA ZNAKÓW.
    
    NSRange subrange = {0, glyphNum};
    NSAttributedString *newString = [string attributedSubstringFromRange:subrange];
    
	// Ponowne utworzenie linii i tablicy przebiegu dla nowego ciągu tekstowego.
	if (glyphNum)
	{
		CFRelease(line);
        
		line = CTLineCreateWithAttributedString((__bridge CFAttributedStringRef)newString);
		if (!line) {NSLog(@"Błąd podczas ponownego utworzenia linii."); return;}
		
		lineLength = CTLineGetTypographicBounds(line, NULL, NULL, NULL);
		runArray = CTLineGetGlyphRuns(line);
	}
 
	// Przechowywanie informacji o odległości znaku od początku ścieżki.
    // by móc obliczyć procentową odległość danego punktu na ścieżce.
	float glyphDistance = 0.0f;
		
    CGContextRef context = UIGraphicsGetCurrentContext();
    CGContextSaveGState(context);
    
    // Ustalenie pozycji początkowych.
    CGPoint textPosition = CGPointMake(0.0f, 0.0f);
	CGContextSetTextPosition(context, textPosition.x, textPosition.y);
    
    for (CFIndex runIndex = 0; runIndex < CFArrayGetCount(runArray); runIndex++) 
	{
		// Pobranie przebiegu.
		CTRunRef run = (CTRunRef)CFArrayGetValueAtIndex(runArray, runIndex);
		
        // Pobranie czcionki i koloru.
		CFDictionaryRef attributes = CTRunGetAttributes(run);
		CTFontRef runFont = CFDictionaryGetValue(attributes, kCTFontAttributeName);
		CGColorRef fontColor = (CGColorRef) CFDictionaryGetValue(attributes, kCTForegroundColorAttributeName);
		CFShow(attributes);
		if (fontColor) [[UIColor colorWithCGColor:fontColor] set];
		
		// Iteracja przez każdy znak w przebiegu.
		for (CFIndex runGlyphIndex = 0; runGlyphIndex < CTRunGetGlyphCount(run); runGlyphIndex++) 
		{
			// Obliczenie procentowej odległości od początku ścieżki.
			float glyphWidth = CTRunGetTypographicBounds(run, CFRangeMake(runGlyphIndex, 1), NULL, NULL, NULL);			
			float percentConsumed = glyphDistance / lineLength;
            
			// Wyszukanie na ścieżce odpowiedniej pary punktów.
			CFIndex index = 1;
			while ((index < pointPercentArray.count) && 
				   (percentConsumed > [[pointPercentArray objectAtIndex:index] floatValue]))
				index++;
			
			// W przypadku braku danych nie próbuj nic wyświetlić. To się nie powinno zdarzyć.
			if (index > (points.count - 1)) continue;
			
			// Obliczenie punktu pośredniego pomiędzy wybraną parą punktów.
			CGPoint point1 = [[points objectAtIndex:index - 1] CGPointValue];
			CGPoint point2 = [[points objectAtIndex:index] CGPointValue];
            
			float percent1 = [[pointPercentArray objectAtIndex:index - 1] floatValue];
			float percent2 = [[pointPercentArray objectAtIndex:index] floatValue];
			float percentOffset = (percentConsumed - percent1) / (percent2 - percent1);
            
			float dx = point2.x - point1.x;
			float dy = point2.y - point1.y;
			
			CGPoint targetPoint = CGPointMake(point1.x + (percentOffset * dx), (point1.y + percentOffset * dy));
			targetPoint.y = view.bounds.size.height - targetPoint.y;
            
			// Określenie przesunięcia x i y w kontekście.
			CGContextTranslateCTM(context, targetPoint.x, targetPoint.y);
			CGPoint positionForThisGlyph = CGPointMake(textPosition.x, textPosition.y);
			
			// Określenie przesunięcia x i y w kontekście.
			float angle = -atan(dy / dx);
			if (dx < 0) angle += M_PI; // Jeśli tekst jest wyświetlany w lewo, trzeba odwrócić znak.
			CGContextRotateCTM(context, angle);
			
			// Zastosowanie przekształcenia na podstawie macierzy.
			textPosition.x -= glyphWidth;
			CGAffineTransform textMatrix = CTRunGetTextMatrix(run);
			textMatrix.tx = positionForThisGlyph.x;
			textMatrix.ty = positionForThisGlyph.y;
			CGContextSetTextMatrix(context, textMatrix);
			
			// Wyświetlenie znaku.
			CGGlyph glyph;
			CGPoint position;
			CGFontRef cgFont = CTFontCopyGraphicsFont(runFont, NULL);
			CFRange glyphRange = CFRangeMake(runGlyphIndex, 1);
			CTRunGetGlyphs(run, glyphRange, &glyph);
			CTRunGetPositions(run, glyphRange, &position);
			CGContextSetFont(context, cgFont);
			CGContextSetFontSize(context, CTFontGetSize(runFont));
			CGContextShowGlyphsAtPositions(context, &glyph, &position, 1);
			
			CFRelease(cgFont);
			
			// Wyzerowanie transformacji kontekstu.
			CGContextRotateCTM(context, -angle);
			CGContextTranslateCTM(context, -targetPoint.x, -targetPoint.y);
			
			glyphDistance += glyphWidth;
		}
	}
	
	CFRelease(line);
	CGContextRestoreGState(context);
}

#define VALUE(_INDEX_) [NSValue valueWithCGPoint:points[_INDEX_]]

// _getPointsFromBezier
void _getPointsFromBezier(void *info, const CGPathElement *element) 
{
    NSMutableArray *bezierPoints = (__bridge NSMutableArray *)info;    
    
    // Pobranie typu elementu ścieżki i jego punktów.
    CGPathElementType type = element->type;
    CGPoint *points = element->points;
    
    // Dodanie punktów, o ile są dostępne  (dla poszczególnych typów).
    if (type != kCGPathElementCloseSubpath)
    {
        [bezierPoints addObject:VALUE(0)];
        if ((type != kCGPathElementAddLineToPoint) &&
            (type != kCGPathElementMoveToPoint))
            [bezierPoints addObject:VALUE(1)];
    }    
    if (type == kCGPathElementAddCurveToPoint)
        [bezierPoints addObject:VALUE(2)];
}

NSArray *_pointsFromBezierPath(UIBezierPath *bpath)
{
    NSMutableArray *points = [NSMutableArray array];
    CGPathApply(bpath.CGPath, (__bridge void *)points, _getPointsFromBezier);
    return points;
}

- (void) drawOnBezierPath: (UIBezierPath *) path
{
    NSArray *points = _pointsFromBezierPath(path);
    [self drawOnPoints:points];
}

@end
